;**************************************************************************
; AudioDDS.ASM

CPU_TYPE equ   2520			  ; or 4520	must also change project property!!
USEPLL	 equ   1			  ; 1=enable PLL for DDS.Set.Freq
UseBSR	 equ   1			  ; for use when BSR should be used instead of Access RAM
FREQ_SIZE equ  3			  ; frequency values are stored in 3 bytes

delay macro delay_time
   movlw	delay_time
   call		Delay.Wmsec
   endm


   if CPU_TYPE == 2520
#include p18f25K20.inc
        LIST P=18f25k20, R=DEC
LCD_DATA	   equ PORTA
LCD_DATA_TRIS  equ TRISA
LCD_CONTROL	   equ PORTB
LCD_CONTROL_TRIS equ TRISB
LCD_ENA		   equ 3
LCD_RS		   equ 4
   else
#include p18f45K20.inc
        LIST P=18f45k20, R=DEC
; the following are for debugging with the PICkit add-on board
LCD_DATA	   equ PORTD
LCD_DATA_TRIS  equ TRISD
LCD_CONTROL	   equ PORTD
LCD_CONTROL_TRIS equ TRISD
LCD_ENA		   equ 2
LCD_RS		   equ 3
   endif


   if 0
	  Port assignments for the 18F25K20
PortA
   0  JoyStick-up
   1  JoyStick-down
   2  JoyStick-left
   3  JoyStick-right
   4  LCD DB4
   5  LCD DB5
   6  LCD DB6
   7  LCD DB7
PortB
   0  QE-I
   1  QE-Q
   2  QE-Enter
   3  LCD-Enable
   4  LCD-RegisterSelect
   5  JoyStick-Enter
   6  ICD
   7  ICD
PortC
   0  LED 0
   1  LED 1
   2  LED 2
   3  SPI clock
   4  nu
   5  SPI Data Out
   6  SPI /CS
   7  Marker
   endif

; some hardware definitions

JS_PORT	 equ   PORTA
JS_UP	 equ   0
JS_DOWN	 equ   1
JS_LEFT	 equ   2
JS_RIGHT equ   3
JS_MASK	 equ   0x0F

QE_PORT	 equ   PORTB
QE_I	 equ   0
QE_Q	 equ   1

ENTER_PORT equ PORTB
QE_ENTER equ   2
JS_ENTER equ   5
ENTER_MASK equ b'00100100'

UP		 equ   1
DOWN	 equ   2
LEFT	 equ   3
RIGHT	 equ   4
ENTER	 equ   5
USE_QE	 equ   6

LED_PORT equ   PORTC
LED0	 equ   0
LED1	 equ   1
LED2	 equ   2

SPI_CSPORT equ   PORTC
SPI_CS	 equ   6

MARKER_PORT equ	  PORTC
MARKER_BIT	equ	  7

; Sweep Types
SWEEP_TYPE_LINEAR   equ	 0
SWEEP_TYPE_OCTAVE1  equ	 1
SWEEP_TYPE_OCTAVE2  equ	 2
SWEEP_TYPE_OCTAVE3  equ	 3
SWEEP_TYPE_OCTAVE4  equ	 4


; set configuration bits
   CONFIG FOSC = INTIO67, FCMEN = OFF, IESO = OFF					; CONFIG1H
   CONFIG PWRT = OFF, BOREN = SBORDIS, BORV = 30					; CONFIG2L
   CONFIG WDTEN = OFF, WDTPS = 32768								; CONFIG2H
   CONFIG CCP2MX = PORTC, PBADEN = OFF, LPT1OSC = OFF, HFOFST = ON 	; CONFIG3H
   CONFIG MCLRE = ON												; CONFIG3H
   CONFIG STVREN = ON, LVP = OFF, XINST = OFF						; CONFIG4L
   CONFIG CP0 = OFF, CP1 = OFF, CP2 = OFF, CP3 = OFF				; CONFIG5L
   CONFIG CPB = OFF, CPD = OFF										; CONFIG5H
   CONFIG WRT0 = OFF, WRT1 = OFF, WRT2 = OFF, WRT3 = OFF			; CONFIG6L
   CONFIG WRTB = OFF, WRTC = OFF, WRTD = OFF						; CONFIG6H
   CONFIG EBTR0 = OFF, EBTR1 = OFF, EBTR2 = OFF, EBTR3 = OFF		; CONFIG7L
   CONFIG EBTRB = OFF												; CONFIG7H

ScratchPadRam   udata_acs   0x10
; the Access RAM ends at 0x5F ==> .96 bytes
; for ISR
W_save		   res	 1
STATUS_save	   res	 1

; variables used by specific functions

FrequencyText  res	 6

;	  Delay functions
DelayCount_1   res	 1
DelayCount_2   res	 1
DelayCount_3   res	 1
DelayCount_4   res	 1
Delay_100val   res	 1

DDS_Freq	   res	 FREQ_SIZE

;	  LCD
Nibble		   res	 1
LCD_Value	   res	 1		  ; LCD_Cmd, LCD_Data, LCD_Byte
LCD_col_nbr	   res	 1		  ; cursor location: 0 relative
LCD_line_nbr   res	 1		  ; cursor location: 0 relative

MenuAddr	   res	 2

SET_DDS_FREQ   equ	 0		  ; bit 0 of Flags byte
NOT_FREQUENCY  equ	 1
MANUAL_STEP	   equ	 2		  ; manual linear sweep
Flags	 	   res	 1

Temp_List_Count res	 1
List_Freqs_Addr res	 1

; Enter.Value
LS_digit_pos   res	 1
Cur_digit_pos  res	 1
MS_digit_pos   res	 1

;	  Mpy.10, Asr.3, Asl.3, Add.3, Sub.3
Accumulator	   res	 4
AccumTemp	   res	 4
Addend		   res	 4

; misc variables
QD_count	   res	 1
LoopCount	   res	 1

Sweep_Steps	   res	 2
Sweep_Steps_Loop res 2
Dur_Loop	   res	 2		  ; used for both Sweep and List

Addend_save	   res	 1
EV_FSR2		   res	 1
Nbr_Digits	   res	 1		  ; nbr of digits to print

; intermediate values for Octave sweep
Octave_Start   res	 FREQ_SIZE
Octave_Inc	   res	 FREQ_SIZE
Octave_Count   res	 1

; menu selection - used when returning from selection
SwitchValue	   res	 1
SwitchTemp	   res	 1
Main_Menu_Sel  res	 1
DoSave_Sel	   res	 1
DoList_Sel	   res	 1
DoSweep_Sel	   res	 1
DoSweep.Fstep_Sel res 1

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;						BSR = 0
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

LCD_EEPROM_data udata  0x060  ; remaining bytes of the 0th RAM block

; NOTE: these values must always be accessed via either BSR=0 or an FSR
; MOST accesses are via an FSR

; DDS values from EEPROM - these are binary values
; the frequencies are 1 Hz resolution, in binary and are Little Endian
MAX_LIST_COUNT equ	 30
EEPROM_data
DDS_Flag	   res	 1		  ; EEPROM_SIZE = valid data in EEPROM
Startup_Function res 1
Single_Freq	   res	 FREQ_SIZE
Sweep_Fhi	   res	 FREQ_SIZE
Sweep_Flo	   res	 FREQ_SIZE
Sweep_Fstep	   res	 FREQ_SIZE
Sweep_Marker   res	 FREQ_SIZE
Sweep_Type	   res	 1
Sweep_Dur	   res	 2
List_Count	   res	 1
List_Dur	   res	 2
List_Freqs	   res	 MAX_LIST_COUNT*FREQ_SIZE
EEPROM_SIZE	   equ	 $ - EEPROM_data

LCD_line	   res	  41	  ; working lines for LCD text
BSRpad		   res	  1
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;						Interrupt Vectors
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  org   0
  goto  Main

  org   0x08
  goto  HiPriInt

  org   0x18
  goto  LoPriInt

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;						Constants in low memory
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

PackedData   code_pack	0x20	  ; for strings

;  this table MUST be within the 1st 256 bytes of Flash
LCD_LineOffset	  db	0x80+0x0, 0x80+0x40, 0x80+0x14, 0x80+0x54
DDS_Default		  db	0x90, 0x01, 0, 0	 ; 400Hz is default freq.

; Addend values for modifying frequency
One				  db	1,0,0,0
Ten				  db	10, 0,0,0
OneHundred		  db	100, 0,0,0
OneThousand		  db	0xE8, 3, 0,0
TenThousand		  db	0x10, 0x27, 0,0

Txt_Title	   db "   PIC Audio DDS\n"
			   db "       K3PTO", 0
; The menus are limited to three lines of up to two items each
; Each item must be eight or fewer characters each
; The items must begin in row 1 (0 relative) and start in columns 1 and 11
; The allowable starting positions are:
;		row column	      row  column
;		 1	   1		   1	 11
;		 2	   1		   2	 11
;		 3	   1		   3	 11
; The value reported back by the function Do.Menu is based on the cursor position
; of the selected menu item.  The cursor will always be in the column preceding
; the first character of the item.
; The formula for the return value is Column + Row of the cursor as follows:
;		 01	   11
;		 02	   12
;		 03	   13

Main_Menu	   db "\fMain Menu\n"
			   db  " SingleF   Save\n"
			   db  " Sweep\n"
			   db  " List", 0
MAIN_MENU_SINGLE  equ	1
MAIN_MENU_SAVE	  equ	11
MAIN_MENU_SWEEP	  equ	2
MAIN_MENU_LIST	  equ	3

Menu_arrow	   db ">\b", 0
Menu_erase	   db " \b", 0
Txt_Frequency  db "\fFrequency  ", 0

Sweep_Menu	   db "\fSweep Menu\n"
			   db  " FLow      FHigh\n"
			   db  " Step      Dur\n"
			   db  " Marker    GO", 0
SWEEP_MENU_FLOW	  equ	1
SWEEP_MENU_FHIGH  equ	11
SWEEP_MENU_STEP	  equ	2
SWEEP_MENU_DUR	  equ	12
SWEEP_MENU_MARKER equ	3
SWEEP_MENU_GO	  equ	13

Step_Menu	   db "\fStep Size Menu\n"
			   db  " Fixed\n"
			   db  " Octave    Octave/2\n"
			   db  " Octave/3  Octave/4", 0
STEP_MENU_FIXED	  equ	1
STEP_MENU_OCT	  equ	2
STEP_MENU_OCT_2	  equ	12
STEP_MENU_OCT_3	  equ	3
STEP_MENU_OCT_4	  equ	13


TxtSweepFlow   db "\fSweep F Low  ", 0
TxtSweepFhigh  db "\fSweep F High  ", 0
TxtSweepFstep  db "\fSweep F Step  ", 0
TxtSweepDur	   db "\fStep Dur ", 0
TxtSweepMarker db "\fSweep Marker ", 0
TxtSweepRun	   db "\fSweep Freq: ", 0

List_Menu	   db "\fFrequency List Menu\n"
			   db  " Count     Dur \n"
			   db  " Freqs     Go ", 0
LIST_MENU_COUNT	  equ	1
LIST_MENU_DUR	  equ	11
LIST_MENU_FREQS	  equ	2
LIST_MENU_GO	  equ	12

TxtListCount   db "\fCount (<=30) ", 0
TxtListDur	   db "\fDuration (ms) ", 0
TxtListRun	   db "\fList Freq: "\0

Save_Menu	   db "\fSel Startup Option\n"
			   db " MainMenu  SingleF\n"
			   db " Sweep     List\n", 0
				  ;01234567890123456789
SAVE_MENU_MAIN	  equ	1
SAVE_MENU_SINGLE  equ	11
SAVE_MENU_SWEEP	  equ	2
SAVE_MENU_LIST	  equ	12

TxtSaveVerify  db "\rVerifying - ", 0
TxtSaveSuccess db "Success", 0
TxtSaveError   db "Failed", 0

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;						Main Program
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

   code
Main
; clear the first RAM bank
   lfsr		0, 0			  ; start with address 0
   movlw	0				  ; clear 256 bytes
Main.5
   clrf		POSTINC0		  ; clear a byte - using FSR precludes using BSR
   decf		WREG			  ; update bytecount
   bnz		Main.5			  ; br if not done
; some initialization
   call		Init.Ports		  ; initialize the I/O ports
   call		LCD.Init		  ; initialize the LCD
   movlb	0				  ; point to RAM bank 0
   lfsr		0, Txt_Title	  ; print the title screen
   call		Print.ROM
   movlw	10				  ; set up to
   call		Delay.W100msec	  ;		delay 1 second
   call		DDS.Init

; set up the initial menu locations
   movlw	1				  ; row 1, column 0
   movwf	Main_Menu_Sel
   movwf	DoSave_Sel
   movwf	DoList_Sel
   movwf	DoSweep_Sel
   movwf	DoSweep.Fstep_Sel

; check for startup value in EEPROM
   call		Read.EEPROM
   movlw	SAVE_MENU_MAIN
   subwf	Startup_Function, w, UseBSR  ; STARTUP_MENU:
   bz		Main.100		  ; br if yes

   movlw	SAVE_MENU_SINGLE
   subwf	Startup_Function, w, UseBSR ; STARTUP_SINGLE_FREQ?
   bnz		Main.10			  ; br if no
   lfsr		2, Single_Freq	  ; point to the frequency value
   call		DDS.Set.Freq	  ; set the DDS
   call	 	Do.Single.Freq
   bra		Main.100
Main.10
   movlw	SAVE_MENU_SWEEP
   subwf	Startup_Function, w, UseBSR ; STARTUP_SWEEP?
   bnz		Main.20			  ; br if no
   call		Do.Sweep.Go.05
   bra		Main.100
Main.20
   movlw	SAVE_MENU_LIST
   subwf	Startup_Function, w, UseBSR ; STARTUP_LIST?
   bnz		Main.100		  ; br if no
   call		Do.List.Go.05

Main.100
; print main menu and get selection
   movlw	LOW Main_Menu
   movwf	MenuAddr		  ; save menu address
   movlw	HIGH Main_Menu
   movwf	MenuAddr+1
   movff	Main_Menu_Sel, SwitchValue ; get previoius menu selection
   call		Do.Menu			  ; get main function
   bz		$+6		 		  ; br if menu exit was requested
   movff	SwitchValue, Main_Menu_Sel
Main.110
; execute the selection
Try.Single.Freq
   movlw	MAIN_MENU_SINGLE
   subwf	SwitchValue, w	  ; is it SingleF
   bnz		Try.Sweep		  ; br if no
   call		Do.Single.Freq
   bra		Main.100
Try.Sweep
   movlw	MAIN_MENU_SWEEP
   subwf	SwitchValue, w
   bnz		Try.List
   call		Do.Sweep
   bra		Main.100
Try.List
   movlw	MAIN_MENU_LIST
   subwf	SwitchValue, w
   bnz		Try.Save
   call		Do.List
   bra		Main.100
Try.Save
   movlw	MAIN_MENU_SAVE
   subwf	SwitchValue, w
   bnz		Main.100
   call		Do.Save
   bra		Main.100


Do.Save
; print menu and get selection
   movlw	LOW Save_Menu
   movwf	MenuAddr		  ; save menu address
   movlw	HIGH Save_Menu
   movwf	MenuAddr+1
   movff	DoSave_Sel, SwitchValue ; get previoius menu selection
   call		Do.Menu			  ; get main function
   bz		$+6		 		  ; br if menu exit was requested
   movff	SwitchValue, DoSave_Sel
; check if STARTUP_MENU
   movlw	SAVE_MENU_MAIN
   cpfseq	SwitchValue
   bra		Do.Save.10		  ; br if not equal
   bra		Do.Save.50
Do.Save.10
; check if STARTUP_SINGLE_FREQ
   movlw	SAVE_MENU_SINGLE
   cpfseq	SwitchValue
   bra		Do.Save.20		  ; br if not equal
   bra		Do.Save.50
Do.Save.20
; check if STARTUP_SWEEP
   movlw	SAVE_MENU_SWEEP
   cpfseq	SwitchValue
   bra		Do.Save.30		  ; br if not equal
   bra		Do.Save.50
Do.Save.30
; check if STARTUP_LIST
   movlw	SAVE_MENU_LIST
   cpfseq	SwitchValue
   return			 		  ; return if no selection
Do.Save.50
   movwf	Startup_Function
   goto		Write.EEPROM	  ; write the EEPROM and return
; Do.Save


Do.List
; print menu and get selection
   movlw	LOW List_Menu
   movwf	MenuAddr		  ; save menu address
   movlw	HIGH List_Menu
   movwf	MenuAddr+1
   movff	DoList_Sel, SwitchValue ; get previoius menu selection
   call		Do.Menu			  ; get main function
   bz		$+6		 		  ; br if menu exit was requested
   movff	SwitchValue, DoList_Sel
   bcf		Flags, SET_DDS_FREQ ; disable setting DDS frequency
; check for valid values
   movlw	0xE0			  ; invalid value mask
   andwf	List_Count, w, UseBSR ; is value valid?
   bz		Do.List.Count	  ; br if yes
   lfsr		0, List_Count	  ; start of List values
   movlw	MAX_LIST_COUNT*3  ; nbr of bytes in frequency list
   addlw	3				  ; adjust for List_Count and List_Dur
Do.List.10
   clrf		POSTINC0		  ; clear a byte
   addlw	-1				  ; update byte count
   bnz		Do.List.10		  ; br if not done

; execute the selection
Do.List.Count
   movlw	LIST_MENU_COUNT	  ; is it Count?
   cpfseq	SwitchValue		  ; skip if yes
   bra		Do.List.Dur
Do.List.Count.10
;   set up for call to Do.Single.Freq.1
   lfsr		2, DDS_Freq+2
   clrf		POSTDEC2		  ; clear byte 2
   clrf		POSTDEC2		  ; clear byte 1
   movf		List_Count, w	  ; get current count
   bnz		Do.List.Count.20  ; br if valid value
   incf		List_Count		  ; force to 1
Do.List.Count.20
   movff	List_Count, INDF2 ; copy count
   lfsr		0, TxtListCount	  ; address of count text
   bsf		Flags, NOT_FREQUENCY ; disallow setting DDS frequency
   call		Do.Single.Freq.1  ; get new count value
   bcf		Flags, NOT_FREQUENCY ; allow setting DDS frequency
; check for valid value
   movff	INDF2, List_Count ; save the count value
   movlw	MAX_LIST_COUNT
   cpfsgt	List_Count, UseBSR ; skip if invalid value
   bra		Do.List
; handle invalid value
   movlw	MAX_LIST_COUNT	  ; force max
   movwf	List_Count, UseBSR ;		value
   bra		Do.List.Count.10

Do.List.Dur
   movlw	LIST_MENU_DUR	  ; is it Duration?
   cpfseq	SwitchValue		  ; skip if no
   bra		Do.List.Freq
; set up for call to Do.Single.Freq.1
   lfsr		2, DDS_Freq+2
   clrf		POSTDEC2		  ; clear byte 2
   movff	List_Dur+1, POSTDEC2 ; copy current
   movff	List_Dur, INDF2	  ;		   duration value
   lfsr		0, TxtListDur	  ; address of text
   bsf		Flags, NOT_FREQUENCY ; disallow setting DDS frequency
   call		Do.Single.Freq.1  ; get new count value
   bcf		Flags, NOT_FREQUENCY ; allow setting DDS frequency
; save the value
   movff	POSTINC2, List_Dur
   movff	INDF2, List_Dur+1
   bra		Do.List

Do.List.Freq
   movlw	LIST_MENU_FREQS
   cpfseq	SwitchValue		  ; is it Frequency List?
   bra		Do.List.Go		  ; br if no
   lfsr		2, List_Freqs	  ; point to freq list
   movff	List_Count, Temp_List_Count ; working copy of list count
   bcf		Flags, SET_DDS_FREQ ; disable setting DDS frequency			????
Do.List.Freq.10
   call		Do.Single.Freq.0  ; get new frequency
   dcfsnz	Temp_List_Count	  ; update count
   bra		Do.List			  ; exit if Temp_List_Count=0
   movlw	FREQ_SIZE		  ; set up to
   addwf	FSR2L, f		  ; update pointer to frequency list
   bnc		Do.List.Freq.10
   incf		FSR2H			  ; update upper bits if carry from low byte
   bra		Do.List.Freq.10	  ; get next frequency

Do.List.Go
   movlw	LIST_MENU_GO	  ; test for go
   cpfseq	SwitchValue		  ; skip if yes
   return
   bcf		Flags, MANUAL_STEP ; initial disable manual step
Do.List.Go.05
   movff	List_Count, Temp_List_Count ; working copy of list count
   lfsr		2, List_Freqs	  ; point to freq list
; set up display
   lfsr		0, TxtListRun	  ; point to title text
   call		Print.ROM
   movff	LCD_col_nbr, MS_digit_pos ; save position to print frequency
   bsf		Flags, SET_DDS_FREQ ; enable setting DDS frequency
Do.List.Go.10
   call		Show.Freq		  ; set DDS and display frequency
   btfsc	Flags, MANUAL_STEP ; skip if NOT in manual step mode
   bra		Do.List.Go.30
; delay for Sweep_Dur milliseconds
   movff	List_Dur, Dur_Loop ; working copy
   movff	List_Dur+1, Dur_Loop+1 ; of duration
   call		Wait.Dur
   bn		Do.List.Go.20	  ; br if no switch active
   movlw	ENTER
   cpfseq	SwitchTemp		  ; skip if it was ENTER?
   bra		Do.List			  ; not ENTER - show List menu
   bsf		Flags, MANUAL_STEP ; set up manual step mode
Do.List.Go.20
; update count and list pointer
   dcfsnz	Temp_List_Count
   bra		Do.List.Go.05	  ; start over
   movlw	FREQ_SIZE		  ; set up to
   addwf	FSR2L, f		  ; update pointer to frequency list
   bnc		Do.List.Go.10
   incf		FSR2H			  ; update upper bits if carry from low byte
   bra		Do.List.Go.10	  ; get next frequency
Do.List.Go.30
; handle switch
   call		Wait.Switch		  ; wait for another switch
   call		Switch.Off
   movlw	RIGHT			  ; test for joystick right
   subwf	SwitchValue, w	  ; is it?
   bz		Do.List.Go.20	  ; br if yes - do next higher frequency
   movlw	LEFT			  ; test for joystick left
   subwf	SwitchValue, w	  ; is it?
   bz		Do.List.Go.35	  ; br if yes - step backwards
   movlw	ENTER			  ; test for enter = resume
   subwf	SwitchValue, w	  ; is it?
   bnz		Do.List.Go.30	  ; br if no - error
   bcf		Flags, MANUAL_STEP ; disable manual step mode
   bra		Do.List.Go.20	  ; procede with normal sweep
Do.List.Go.35
; update count and list pointer for backing up one frequency
   incf	 	Temp_List_Count
   movlw	FREQ_SIZE		  ; set up to
   subwf	FSR2L, f		  ;		update low byte of pointer to frequency list
   movlw	0				  ; set up to
   subwfb	FSR2H			  ;		update high byte
   bra		Do.List.Go.10	  ; get next frequency
; Do.List


Show.Freq
; display the frequency and set the DDS
; enter with
;	  FSR2 = address of frequency
;	  MS_digit_pos = LCD column nbr for MS digit

   bsf		Flags, SET_DDS_FREQ ; enable setting DDS frequency
   call		DDS.Set.Freq	  ; set DDS frequency
   lfsr		1, DDS_Freq		  ; address of frequency
   lfsr		0, FrequencyText+5 ; addr to put trailing NULL
   movff	MS_digit_pos, LCD_col_nbr ; place
   call		LCD.Pos			  ; LCD cursor
   goto		Int.2.Text		  ; print the frequency and return
; Show.Freq


Do.Single.Freq  ; allow user to modify displayed 3-byte value
   lfsr		2, Single_Freq	  ; point to frequency

Do.Single.Freq.0   ; enter here with FSR2 = address of value to update
   lfsr		0, Txt_Frequency  ; point to frequency text in ROM

Do.Single.Freq.1	   ; enter here with FSR2 = address of value to update
			   ; and FSR0 = address of entry text
   call		Print.ROM		  ; print text title
   movff	LCD_col_nbr, MS_digit_pos ; save position to print frequency
   lfsr		0, FrequencyText+5 ; addr to put trailing NULL
   call		Int.2.Text		  ; convert binary to text
   movlw	'\b'			  ; back up
   call		LCD.PrintW		  ;		to LSdigit
   movff	LCD_col_nbr, LS_digit_pos ; save position of LSdigit
   call		Enter.Value		  ; allow user to enter desired frequency
   goto		DDS.Set.Freq	  ; set the frequency and return to caller
; Do.Single.Freq


Do.Sweep
   movlw	LOW Sweep_Menu
   movwf	MenuAddr		  ; save menu address
   movlw	HIGH Sweep_Menu
   movwf	MenuAddr+1
   movff	DoSweep_Sel, SwitchValue ; get previoius menu selection
   call		Do.Menu			  ; get sweep function
   bz		$+6		 		  ; br if menu exit was requested
   movff	SwitchValue, DoSweep_Sel
; test for valid values read from EEPROM
;	  if Sweep_Flo is negative,, then data is not valid
   btfss	Sweep_Flo+2, 7, UseBSR ; check for valid MSByte
   bra		Do.Sweep.FLow	  ; br if valid
; insert 400Hz into FLow, FHigh and Fstep, clear Duration
   movlw	FREQ_SIZE
   lfsr		0, DDS_Default
   lfsr		1, Sweep_Flo	  ; point to F low
   call		ROM.2.RAM
   movlw	FREQ_SIZE
   lfsr		1, Sweep_Fhi	  ; address to copy to
   call		ROM.2.RAM
   movlw	FREQ_SIZE
   lfsr		1, Sweep_Fstep	  ; address to copy to
   call		ROM.2.RAM
   movlw	FREQ_SIZE
   lfsr		1, Sweep_Marker	  ; address to copy to
   call		ROM.2.RAM
   clrf		Sweep_Dur, 1	  ; clear the duration value
   clrf		Sweep_Dur+1, 1

Do.Sweep.FLow
   movlw	SWEEP_MENU_FLOW	  ; test for Flo
   cpfseq	SwitchValue		  ; br if yes
   bra		Do.Sweep.FHigh
;   set up for call to Do.Single.Freq.1
   lfsr		2, Sweep_Flo	  ; address of current Flo
   lfsr		0, TxtSweepFlow	  ; point to title text
   call		Do.Single.Freq.1  ; get the Low frequency value
   bra		Do.Sweep

Do.Sweep.FHigh
   movlw	SWEEP_MENU_FHIGH  ; test for Fhi
   cpfseq	SwitchValue		  ; br if yes
   bra		Do.Sweep.Fstep
;   set up for call to Do.Single.Freq
   lfsr		2, Sweep_Fhi	  ; address of current Fhi
   lfsr		0, TxtSweepFhigh  ; point to title text
   call		Do.Single.Freq.1  ; get the High frequency value
   bra		Do.Sweep

Do.Sweep.Fstep
   bcf		Flags, SET_DDS_FREQ ; disable setting DDS frequency
   movlw	SWEEP_MENU_STEP	  ; test for Step definition
   cpfseq	SwitchValue		  ; skip if yes
   bra		Do.Sweep.Dur
; show step menu and get operator input
   movlw	LOW Step_Menu
   movwf	MenuAddr		  ; save menu address
   movlw	HIGH Step_Menu
   movff	DoSweep.Fstep_Sel, SwitchValue ; get previoius menu selection
   movwf	MenuAddr+1
   call		Do.Menu			  ; get step function
   bz		$+6		 		  ; br if menu exit was requested
   movff	SwitchValue, DoSweep.Fstep_Sel
; check for linear sweep
   movlw	STEP_MENU_FIXED	  ; test for Fstep
   cpfseq	SwitchValue		  ; skip if yes
   bra		Do.Sweep.Step.Octave1
;   set up for call to Do.Single.Freq
   movlw	SWEEP_TYPE_LINEAR ; show linear sweep
   movwf	Sweep_Type		  ; save it
   lfsr		2, Sweep_Fstep	  ; address of current Fstep
   lfsr		0, TxtSweepFstep  ; point to title text
   bsf		Flags, NOT_FREQUENCY ; disallow setting DDS frequency
   call		Do.Single.Freq.1  ; get the frequency step value
   bcf		Flags, NOT_FREQUENCY ; allow setting DDS frequency
   bra		Do.Sweep
Do.Sweep.Step.Octave1
   movlw	STEP_MENU_OCT	  ; test for SWEEP_TYPE_OCTAVE1
   cpfseq	SwitchValue		  ; skip if yes
   bra		Do.Sweep.Step.Octave2
   movlw	SWEEP_TYPE_OCTAVE1
   movwf	Sweep_Type		  ; save it
   bra		Do.Sweep
Do.Sweep.Step.Octave2
   movlw	STEP_MENU_OCT_2	  ; test for SWEEP_TYPE_OCTAVE2
   cpfseq	SwitchValue		  ; skip if yes
   bra		Do.Sweep.Step.Octave3
   movlw	SWEEP_TYPE_OCTAVE2
   movwf	Sweep_Type		  ; save it
   bra		Do.Sweep
Do.Sweep.Step.Octave3
   movlw	STEP_MENU_OCT_3	  ; test for SWEEP_TYPE_OCTAVE3
   cpfseq	SwitchValue		  ; skip if yes
   bra		Do.Sweep.Step.Octave4
   movlw	SWEEP_TYPE_OCTAVE3
   movwf	Sweep_Type		  ; save it
   bra		Do.Sweep
Do.Sweep.Step.Octave4
   movlw	STEP_MENU_OCT_4	  ; test for SWEEP_TYPE_OCTAVE4
   cpfseq	SwitchValue		  ; skip if yes
   bra		Do.Sweep
   movlw	SWEEP_TYPE_OCTAVE4
   movwf	Sweep_Type		  ; save it
   bra		Do.Sweep

Do.Sweep.Dur
   movlw	SWEEP_MENU_DUR	  ; test for duration
   cpfseq	SwitchValue		  ; br if yes
   bra		Do.Sweep.Marker
; set up for call to Do.Single.Freq.1
   lfsr		2, Sweep_Dur
   lfsr		0, TxtSweepDur	  ; point to title text
   bsf		Flags, NOT_FREQUENCY ; disallow setting DDS frequency
   call		Do.Single.Freq.1  ; get the duration
   bcf		Flags, NOT_FREQUENCY ; allow setting DDS frequency
   bra		Do.Sweep

Do.Sweep.Marker
   movlw	SWEEP_MENU_MARKER ; test for marker
   cpfseq	SwitchValue		  ; br if yes
   bra		Do.Sweep.Go
;   set up for call to Do.Single.Freq
   lfsr		2, Sweep_Marker	  ; address of current Marker
   lfsr		0, TxtSweepMarker ; point to title text
   call		Do.Single.Freq.1  ; get the Marker frequency value
   bra		Do.Sweep

Do.Sweep.Go
   movlw	SWEEP_MENU_GO	  ; test for go
   cpfseq	SwitchValue		  ; skip if yes
   return
Do.Sweep.Go.05
   bsf		Flags, SET_DDS_FREQ ; enable setting DDS frequency
; set up display
   lfsr		0, TxtSweepRun	  ; point to title text
   call		Print.ROM
   movff	LCD_col_nbr, MS_digit_pos ; save position to print frequency
; check for linear sweep request
   movlw	SWEEP_TYPE_LINEAR
   cpfseq	Sweep_Type
   bra		Do.Sweep.Octave
;calculate the number of steps in the range
   lfsr		0, Sweep_Fstep	  ; copy
   lfsr		1, Accumulator	  ;	 Fstep
   call		Copy.Freq.Bytes	  ;		to Accumulator
   lfsr		1, Accumulator	  ; divide
   call		Asr.3			  ;		Fstep by 2
   lfsr		0, Accumulator	  ; Accumulator =
   lfsr		1, Sweep_Fhi	  ;	 Fhigh +
   call		Add.3			  ;		Fstep/2
   lfsr		1, Sweep_Flo	  ; Accumulator =
   call		Sub.3			  ;		Accumulator - Flow
   lfsr		0, Sweep_Fstep	  ; copy
   lfsr		1, AccumTemp	  ;	 Fstep
   call		Copy.Freq.Bytes	  ;		to AccumTemp
   call		FXD2416U		  ; Accumulator = (Fhigh + Fstep/2 - Flow)/Fstep
   movff	Accumulator, Sweep_Steps ; save
   movff	Accumulator+1, Sweep_Steps+1 ; nbr of steps
   infsnz	Sweep_Steps		  ; make "inclusive"
   incf		Sweep_Steps+1	  ; update high byte if low byte overflowed to 0
   bcf		Flags, MANUAL_STEP ; initial disable manual step

Do.Sweep.Go.1
; initialize for start of sweep
   movff	Sweep_Steps, Sweep_Steps_Loop ; working copy
   movff	Sweep_Steps+1, Sweep_Steps_Loop+1 ;  of nbr of steps
;  working copy of frequency that is modified during sweep
   lfsr		0, Sweep_Flo	  ; address of F low
   lfsr		1, DDS_Freq		  ;	 copy to
   call		Copy.Freq.Bytes	  ;		DDS Frequency for DDS.Set.Freq
Do.Sweep.Go.10
   call		Check.Marker	  ; check if marker frequency
   lfsr		2, DDS_Freq		  ; point to frequency bytes for Show.Freq
   call		Show.Freq		  ; set DDS and display frequency
   btfsc	Flags, MANUAL_STEP ; skip if NOT in manual step mode
   bra		Do.Sweep.Go.12
; delay for Sweep_Dur milliseconds
   movff	Sweep_Dur,Dur_Loop ; working copy
   movff	Sweep_Dur+1,Dur_Loop+1 ; of duration
   call		Wait.Dur
   bn		Do.Sweep.Go.20	  ; br if no switch active
   movlw	ENTER
   cpfseq	SwitchTemp		  ; was it ENTER?
   bra		Do.Sweep		  ; back to List menu
   bsf		Flags, MANUAL_STEP ; set up manual step mode
Do.Sweep.Go.12
; handle switch
   call		Wait.Switch		  ; wait for another switch
   call		Switch.Off
   movlw	RIGHT			  ; test for joystick right
   subwf	SwitchValue, w	  ; is it?
   bz		Do.Sweep.Go.20	  ; br if yes - do next higher frequency
   movlw	LEFT			  ; test for joystick left
   subwf	SwitchValue, w	  ; is it?
   bz		Do.Sweep.Go.15	  ; br if yes - step backwards
   movlw	ENTER			  ; test for joystick enter = resume
   subwf	SwitchValue, w	  ; is it?
   bnz		Do.Sweep.Go.12	  ; br if no - error
   bcf		Flags, MANUAL_STEP ; disable manual step mode
   bra		Do.Sweep.Go.20	  ; procede with normal sweep
; handle backwards step
Do.Sweep.Go.15
   incf		Sweep_Steps_Loop  ; update LSByte of counter
   bnz		Do.Sweep.Go.17
   incf		Sweep_Steps_Loop+1 ; update MSByte
Do.Sweep.Go.17
   lfsr		0, DDS_Freq
   lfsr		1, Sweep_Fstep
   call		Sub.3			  ; DDS_Freq = DDS_Freq - Sweep_Fstep
   bra		Do.Sweep.Go.10
Do.Sweep.Go.20
; set up for next higher frequency
   decf		Sweep_Steps_Loop  ; update LSByte of counter
   bnz		Do.Sweep.Go.21
   decf		Sweep_Steps_Loop+1 ; update MSByte
   bn		Do.Sweep.Go.1	  ; br if done all steps
Do.Sweep.Go.21
   lfsr		0, DDS_Freq
   lfsr		1, Sweep_Fstep
   call		Add.3			  ; DDS_Freq = DDS_Freq + Sweep_Fstep
   bra		Do.Sweep.Go.10

Do.Sweep.Octave
;  get initial frequency for sweep
   lfsr		0, Sweep_Flo	  ; address of starting frequency
   lfsr		1, Octave_Start	  ;	 copy to
   call		Copy.Freq.Bytes	  ;		octave start frequency
Do.Sweep.Octave.10
; calculate octave increment value = Octave_Start/Sweep_Type(=1,2,3,4)
   movff	Sweep_Type, AccumTemp ; get nbr of intervals in octave
   clrf		AccumTemp+1		  ; make 16 bit divisor
   lfsr		0, Octave_Start	  ; set up
   lfsr		1, Accumulator	  ;		to copy
   call		Copy.Freq.Bytes	  ;		   octave start
   call		FXD2416U		  ; to get octave increment value
   lfsr		0, Accumulator	  ; save
   lfsr		1, Octave_Inc	  ;		octave increment
   call		Copy.Freq.Bytes	  ;		   value
; start sweep at Octave_Start
   lfsr		0, Octave_Start	  ; get working
   lfsr		1, DDS_Freq		  ;		copy of
   call		Copy.Freq.Bytes	  ;		   frequency
   movff	Sweep_Type, Octave_Count ; get nbr of intervals in octave
   lfsr		0, Octave_Start	  ; set up for
   call		Asl.3			  ;		next octave starting frequency
Do.Sweep.Octave.20
; check if this was last frequency in sweep by calculating
;	  Sweep_Fhi - DDS_Freq
; the test does not care what the actual subtraction yields other than the
; sign of the result
   movf		DDS_Freq, W		  ; get byte 0 of next frequency
   subwf	Sweep_Fhi, W
   movf		DDS_Freq+1, W	  ; get byte 2 of next frequency
   subwfb	Sweep_Fhi+1, W
   movf		DDS_Freq+2, W	  ; get byte 2 of next frequency
   subwfb	Sweep_Fhi+2, W	  ; if W is negative, then DDS_Freq > Sweep_Fhi
   bn		Do.Sweep.Octave	  ; start sweep over
Do.Sweep.Octave.30
   call		Check.Marker	  ; check if marker frequency
   lfsr		2, DDS_Freq		  ; point to frequency bytes for Show.Freq
   call		Show.Freq		  ; set DDS and display frequency
; delay for Sweep_Dur milliseconds
   movff	Sweep_Dur,Dur_Loop ; working copy
   movff	Sweep_Dur+1,Dur_Loop+1 ; of duration
   call		Wait.Dur
   bn		Do.Sweep.Octave.40  ; br if no switch active
   return
Do.Sweep.Octave.40
   dcfsnz	Octave_Count	  ; update nbr of remaining freqs in octave
   bra		Do.Sweep.Octave.10  ; start next octave
; calculate next frequency
   lfsr		0, DDS_Freq		  ; frequency to update
   lfsr		1, Octave_Inc	  ; update value
   call		Add.3			  ; calculate next frequency in octave
   bra		Do.Sweep.Octave.20
; Do.SweepGo


Check.Marker
; check of DDS_Freq is the same as Sweep_Marker
; if yes - set MARKER_BIT in MARKER_PORT
   bcf		MARKER_PORT, MARKER_BIT ; assume not marker frequency
   movf		Sweep_Marker, w, UseBSR ; get byte0 of marker frequency
   subwf	DDS_Freq, w		  ; match?
   bnz		Check.Marker.10	  ; br if no
   movf		Sweep_Marker+1, w, UseBSR ; get byte1 of marker frequency
   subwf	DDS_Freq+1, w	  ; match?
   bnz		Check.Marker.10	  ; br if no
   movf		Sweep_Marker+2, w, UseBSR ; get byte2 of marker frequency
   subwf	DDS_Freq+2, w	  ; match?
   bnz		Check.Marker.10	  ; br if no
   bsf		MARKER_PORT, MARKER_BIT ; show marker frequency
Check.Marker.10
   return
; Check.Marker


Enter.Value		  ; allow user to enter a value
; enter with
;	  FSR2 = address of current value
;	  MS_digit_pos = LCD column nbr of MSdigit
;	  LS_digit_pos = LCD column nbr of LSdigit
; Exit with
;	  value and displayed text updated
;	  Using Quad Encoder updates frequency "live"
;	  Using Joy Stick will not update frequency until Enter is pressed
; uses
;	  FSR0 and FSR1

; initial set up to modify LSdigit
   movff	FSR2L, EV_FSR2	  ; save address of value
   movlw	LOW One			  ; low addr bits of initial digit increment value
   movwf	Addend_save		  ; save it
   movff	LS_digit_pos, Cur_digit_pos
   clrf		QD_count

; copy digit increment value from ROM to RAM
Enter.Value.5				  ; get the addend from ROM
   bcf		Flags, SET_DDS_FREQ ; disable setting DDS frequency
   movff	Addend_save, TBLPTRL ; retrieve address of digit increment value
   clrf		TBLPTRH
   clrf		TBLPTRU
   lfsr		1, Addend
   movlw	4				  ; each increment value is 4 bytes
Enter.Value.7
   tblrd*+					  ; get a byte from ROM
   movff 	TABLAT, POSTINC1  ; copy it
   decf		WREG			  ; update byte count
   bnz		Enter.Value.7	  ; br if not done

; wait for operator switch entry
Enter.Value.8
   call		Wait.Switch		  ; wait for user input
   delay	255				  ; set up delay

   movff	EV_FSR2, FSR0L	  ; addr of value to update
   lfsr		1, Addend		  ; addr of digit increment value
Enter.Value.UP
   movlw	UP				  ; test for UP
   cpfseq	SwitchValue		  ; skip if it is
   bra		Enter.Value.Down  ; not UP - try DOWN
   call		Add.3			  ; update value

; display updated value
Enter.Value.20
   movff	MS_digit_pos, LCD_col_nbr ; retrieve position to print value
   call		LCD.Pos			  ; place cursor
   lfsr		0, FrequencyText+5 ; address to put text
   call		Int.2.Text		  ; convert value to text and print it
   movff	Cur_digit_pos, LCD_col_nbr ; retrieve position of current digit
   call		LCD.Pos			  ; place cursor
   bra		Enter.Value.5	  ; get next switch entry-must restore Addend!

Enter.Value.Down
   movlw	DOWN			  ; test for DOWN
   cpfseq	SwitchValue		  ; skip if it is
   bra		Enter.Value.Left  ; not DOWN - try LEFT
   call		Sub.3			  ; update value
   bra		Enter.Value.20

Enter.Value.Left
   movlw	LEFT			  ; test for LEFT
   cpfseq	SwitchValue		  ; skip if it is
   bra		Enter.Value.Right  ; not LEFT - try RIGHT
   movf	 	MS_digit_pos, w	  ; position of MSdigit
   subwf	LCD_col_nbr, w	  ; OK to move cursor to more significant digit?
   bz		Enter.Value.8	  ; br if already on MSdigit
   movlw	4				  ; offset to addend for next higher digit
   decf		LCD_col_nbr		  ; point to next higher digit on LCD
Enter.Value.30
   addwf	Addend_save		  ;	update pointer to addend for next digit
   call		LCD.Pos			  ; update cursor
   movff	LCD_col_nbr, Cur_digit_pos ; save position of current digit
   bra		Enter.Value.5

Enter.Value.Right
   movlw	RIGHT			  ; test for RIGHT
   cpfseq	SwitchValue		  ; skip if it is
   bra		Enter.Value.Enter  ; not RIGHT - try ENTER
   movf	 	LS_digit_pos, w	  ; position of LSdigit
   subwf	LCD_col_nbr, w	  ; OK to move cursor to less significant digit?
   bz		Enter.Value.8	  ; br if already on LSdigit
   movlw	-4				  ; offset to addend for next lower digit
   incf		LCD_col_nbr		  ; point to next lower digit on LCD
   bra		Enter.Value.30

Enter.Value.Enter
   movlw	ENTER			  ; test for ENTER
   cpfseq	SwitchValue		  ; skip if it is
   bra		Enter.Value.QD	  ; not ENTER - try quadrature decoder
   call		Switch.Off		  ; wait for Enter release
   bsf		Flags, SET_DDS_FREQ ; enable setting DDS frequency
   goto		DDS.Set.Freq	  ; set the frequency and return to caller

Enter.Value.QD
   movlw	USE_QE			  ; test for quad decoder
   cpfseq	SwitchValue		  ; skip if it is
   bra		Enter.Value.5	  ; not quad decoder - must be error
   bcf		INTCON, INT0IE		; disable QD interrupt
   bsf		Flags, SET_DDS_FREQ ; enable setting DDS frequency
   movf		QD_count		  ; quad decoder count
   bn		Enter.Value.QD.neg ; br if value is negative

Enter.Value.QD.pos
   call		Add.3			  ; update value
   decf		QD_count, f		  ; update counter
   bnz		Enter.Value.QD.pos ; until count = 0
   call		DDS.Set.Freq
   bsf		INTCON, INT0IE		; enable QD interrupt
   bra		Enter.Value.20	  ; display the value and wait for switch

Enter.Value.QD.neg
   call		Sub.3			  ; update value
   incf		QD_count, f		  ; update counter
   bnz		Enter.Value.QD.neg ; until count = 0
   call		DDS.Set.Freq
   bsf		INTCON, INT0IE		; enable QD interrupt
   bra		Enter.Value.20	  ; display the value and wait for switch
; Enter.Value


Do.Menu
; get a menu entry from the operator
; enter with
;	  MenuAddr = address of menu
;		 the menu must be multiple lines, each terminated with a \n
;			   except the last which must be terminated with a \0
;		 line 1 is a title line
;		 each subsequqnt line (3 max) of the menu can have two entries:
;			each entry starts with a space and ends with at least one space
;			entry one starts in column 0
;			entry two starts in column 10
; enter with
;	  SwitchValue = previious selection for this menu
; exit with
;	  SwitchValue = selected menu item number = LCD_col_nbr+LCD_line_nbr
;			1	  11
;			2	  12
;			3	  13
;		 0 = exit request (go left from 1st menu item)

   movff	MenuAddr, FSR0L	  ; retrieve menu address
   movff	MenuAddr+1, FSR0H
   call		Print.ROM		  ; print the menu
Do.Menu.5
;   movlw	1				  ; force
;   movwf	LCD_line_nbr	  ;		 line 1
;   clrf		LCD_col_nbr	  ; force column 0
; retrieve starting row and column for this menu
   clrf		LCD_col_nbr		  ; assume column 0
   movlw	10				  ; set up to test for column 10
   cpfsgt	SwitchValue		  ; skip if column 10
   bra		Do.Menu.7
   movwf	LCD_col_nbr		  ; save for positioning
   subwf	SwitchValue, f	  ; calculate line number
Do.Menu.7
   movff	SwitchValue, LCD_line_nbr ; save line number for positioning

Do.Menu.10
   call		LCD.Pos			  ; place the cursor
   lfsr		0, Menu_arrow	  ; point to the menu arrow character
   call		Print.ROM

Do.Menu.20
   call		Wait.Switch		  ; wait for user
Do.Menu.30
   call		Get.Switches	  ; switch still active?
   bnz		Do.Menu.30		  ;	br if yes
   delay	255

Do.Menu.UP
   movlw	UP				  ; test for UP
   cpfseq	SwitchValue		  ; skip if it is
   bra		Do.Menu.Down		  ; not UP - try DOWN
   movlw	1				  ; make sure valid to go up
   cpfsgt	LCD_line_nbr	  ; skip if yes
   bra		Do.Menu.20		  ; not valid - try again
   call		Erase.Char		  ; erase char at cursor
   decf		LCD_line_nbr	  ; point to previous line
   bra		Do.Menu.10		  ; move the cursor

Do.Menu.Down
   movlw	DOWN			  ; test for DOWN
   cpfseq	SwitchValue		  ; skip if it is
   bra		Do.Menu.Left	  ; not DOWN - try LEFT
   call		Erase.Char		  ; erase char at cursor
   movlw	3				  ; make sure valid to go down
   cpfslt	LCD_line_nbr	  ; skip if yes
   bra		Do.Menu.Right.15  ; go to first entry
   incf		LCD_line_nbr	  ; point to next line
   bra		Do.Menu.10		  ; move the cursor

Do.Menu.Left
   movlw	LEFT  			  ; test for LEFT
   cpfseq	SwitchValue		  ; skip if it is
   bra		Do.Menu.Right	  ; not LEFT - try RIGHT
; check for request to return to previous menu
   call		Erase.Char		  ; erase char at cursor
   tstfsz	LCD_col_nbr		  ; skip next instr if column 0
   bra		Do.Menu.Left.10
   movlw	1				  ; check for line 1 (0 relative)
   cpfseq	LCD_line_nbr	  ; skip if yes
   bra		Do.Menu.Left.10
   clrf		SwitchValue		  ; tell caller exit request
   return
; test for move to previous line
Do.Menu.Left.10
   tstfsz	LCD_col_nbr		  ; skip next instr if column 0
   bra		Do.Menu.Left.20	  ; handle column 10
   movlw	10				  ; set up
   movwf	LCD_col_nbr		  ;		for column 10
   decf		LCD_line_nbr	  ; of previous line
   bra		Do.Menu.10		  ; move the cursor
; move to column 0 of same line
Do.Menu.Left.20
   clrf		LCD_col_nbr		  ; set up for column 0
   bra		Do.Menu.10

Do.Menu.Right
   movlw	RIGHT  			  ; test for RIGHT
   cpfseq	SwitchValue		  ; skip if it is
   bra		Do.Menu.Enter	  ; not RIGHT - try ENTER
;
   call		Erase.Char		  ; erase char at cursor
   tstfsz	LCD_col_nbr		  ; skip if OK to go right
   bra		Do.Menu.Right.10  ; br if in column 0
   movlw	10				  ; point to
   movwf	LCD_col_nbr		  ;		 column 10
   bra		Do.Menu.10		  ; move the cursor
; check if OK to go down one line
Do.Menu.Right.10
   movlw	3				  ; make sure valid to go down
   subwf	LCD_line_nbr, w	  ; w = LCD_line_nbr - w
   bz		Do.Menu.Right.15  ; must be at last entry - go to first entry
   incf		LCD_line_nbr	  ;		of next line
   clrf	 	LCD_col_nbr		  ; point to column 0
   bra		Do.Menu.10		  ; move the cursor
Do.Menu.Right.15
   movlw	1				  ; force
   movwf	LCD_line_nbr	  ;		 line 1
   clrf		LCD_col_nbr		  ; force column 0
   bra		Do.Menu.10

Do.Menu.Enter
   movlw	ENTER  			  ; test for ENTER
   cpfseq	SwitchValue		  ; skip if it is
   bra		Do.Menu	 		  ; not ENTER - must be an error
   call		Switch.Off		  ; wait for release
   movff	LCD_col_nbr, SwitchValue
   movf		LCD_line_nbr, w
   addwf	SwitchValue		  ; create menu entry nbr
   return
; Do.Menu


Copy.Freq.Bytes
   movlw	FREQ_SIZE		  ; nbr of bytes to copy
; drop through to Copy.Bytes

Copy.Bytes	   ; copy bytes, RAM to RAM
; enter with
;	  FSR0 = address of source
;	  FSR1 = address of destination
;	  w = nbr of bytes to copy
; modifies:
;	  FSR0, FSR1, W

   movff	POSTINC0, POSTINC1
   decf	 	WREG
   bnz		Copy.Bytes
   return
; Copy.Bytes


ROM.2.RAM		   ; copy bytes from ROM to RAM
; enter with
;  FSR0 = address of bytes in ROM - must be in low memory (12 bit address)
;  FSR1 = address of destination RAM
;  W = nbr of bytes to copy
; exit with
;	  bytes copied
;	  FSR0 is NOT modified

   movff	FSR0L, TBLPTR	  ; get low byte of address
   movff	FSR0H, TBLPTRH	  ; get high byte of address
   clrf		TBLPTRU
ROM.2.RAM.1
   tblrd*+					  ; get a byte from ROM
   movff	TABLAT,POSTINC1	  ; save it
   decf	 	WREG
   bnz		ROM.2.RAM.1		  ;		until done
   return
; ROM.2.RAM


Int.2.Text		  ; create and print a text string of a 5 digit integer value
; enter with:
;	  FSR0 = address to put trailing NULL byte
;	  FSR2 = LSB address of 3 byte integer
; exit with:
;	  text printed
; modifies
;	  FSR0, Accumulator
; Note: FSR2 is used and restored

; copy value to Accumulator for divide function
   movff	POSTINC2, Accumulator
   movff	POSTINC2, Accumulator+1
   movff	POSTDEC2, Accumulator+2
   mulwf	POSTDEC2		  ; restore
;
   movlw	10				  ; need to divide by 10
   movwf	AccumTemp		  ; save to divisor
   clrf		AccumTemp+1		  ; clear high byte of diviso
   movlw	5				  ; need to do 5 digits
   movwf	SwitchValue		  ; use for loop counter
   clrf		POSTDEC0		  ; insert trailing NULL byte
Int.2.Text.1
   call		FXD2416U		  ; remainder is LSdigit value
   movf		Addend, w		  ; get remiander
   iorlw	'0'				  ; make ASCII digit
   movwf	POSTDEC0		  ; save it
   decfsz	SwitchValue		  ; update byte count
   bra		Int.2.Text.1
   movf		POSTINC0, w		  ; dummy instruction so FSR0 points to MSdigit
   call		LCD.Print
   return
; Int.2.Text

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;						Switch functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


Wait.Dur
; wait Dur_Loop msec OR until any switch is activated and released
; enter with
;	  nbr of milliseconds delay in Dur_Loop
; exit with
;	  N = 1 if done duration
;	  N = 0 if exit due to switch activity
Wait.Dur.14
   decf		Dur_Loop		  ; update LSByte of duration
   bnz		Wait.Dur.16
   decf		Dur_Loop+1		  ; update MSByte of duration
   bn		Wait.Dur.20		  ; br if done duration
Wait.Dur.16
   call		Delay.Msec		  ; delay 1 msec
   call		Get.Switches	  ; check for user terminate request
   bz		Wait.Dur.14		  ; br if no switches activated
   call		Switch.Off		  ; wait for switch release
   bcf		STATUS, N		  ; show switch was active and released
Wait.Dur.20
   return
; Wait.Dur


Wait.Switch		  ; wait for switch or QD change
; exit with:
;	  SwitchValue = UP, DOWN, LEFT, RIGHT, ENTER, USE_QE
;	  if USE_QE, then QD_count

Wait.Switch.1
   call		Get.Switches	  ; any switches active?
   bnz		Wait.Switch.2	  ; br if yes
   movf	 	QD_count, w		  ; QD count?
   bz		Wait.Switch.1	  ; br if no
   movlw	USE_QE			  ; show QD entry
   movwf	SwitchValue		  ; save it for caller
   return
Wait.Switch.2
   movwf	SwitchValue		  ; save the switch value
   delay	50				  ; delay 50msec
   call		Get.Switches	  ; any switches active?
   cpfseq	SwitchValue		  ; same sw still activated?
   bra		Wait.Switch.1	  ; br if no
   return
;Wait.Switch


Get.Switches				  ; read the joystick and check quadrature encodeer
; read the joystick and quadrature encoder Enter switches
; IMPORTANT: switches are low true!
; exit with
;  Z = 1 if no switches
;  Z = 0 if at least one switch is being used
;  w, SwitchTemp = switch number
;			UP, DOWN, LEFT, RIGHT, ENTER

   bcf		STATUS, Z		  ; assume a switch is active
; the following instructions do not affect the Z bit!
   movlw	UP				  ; assume js up
   btfss	JS_PORT, JS_UP
   bra		Get.Switches.Exit
   movlw	DOWN			  ; try down
   btfss	JS_PORT, JS_DOWN
   bra		Get.Switches.Exit
   movlw	LEFT			  ; try left
   btfss	JS_PORT, JS_LEFT
   bra		Get.Switches.Exit
   movlw	RIGHT			  ; try right
   btfss	JS_PORT, JS_RIGHT
   bra		Get.Switches.Exit

; note: the bits are inverted because the "normal" states are high
Get.Switches.Enter
   movlw	0xFF			  ; invert mask
   xorwf	ENTER_PORT, w	  ; read the enter switches and invert the bits
   andlw	ENTER_MASK		  ; leave only the enter bits: Z=1 if no switch
   bz		Get.Switches.Exit1 ; do not update SwitchTemp if no switch active
   movlw	ENTER			  ; assume enter (does not affect Z)
Get.Switches.Exit
   movwf	SwitchTemp		  ; save for Sweep and List - does not affect Z bit
Get.Switches.Exit1
   return
; Get.Switches


Switch.Off	; wait for release
   clrf		LoopCount		  ; init counter for switch debounce
Switch.Off.1
   call		Delay.100usec	  ; delay 100 usec
   call		Get.Switches	  ; check for switch release
   bnz		Switch.Off		  ; br if switch active - start over
   incf		LoopCount		  ; update counter
   movlw	10				  ; must be inactive for 10 iterations
   cpfsgt	LoopCount		  ; skip if LoopCount > w
   bra		Switch.Off.1
   delay	10				  ; wait 10 more msec
   return
; Switch.Off


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;						DDS functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

cs_on macro
   bcf		SPI_CSPORT, SPI_CS
   endm

cs_off macro
   bsf		SPI_CSPORT, SPI_CS
   endm

DDS.Init	; initialize the DDS port and AD9834

; init the serial port
   cs_off
   movlw	b'01000000'		  ; Output data changes on clock transition
   movwf	SSPSTAT			  ;		from active to idle
   movlw	b'00110001'		  ; enable SPI, clock idle high,
   movwf	SSPCON1			  ;		clock freq = Master mode, Fosc/16
   bcf	 	PIR1, SSPIF		  ; init to byte being transmitted
; init the AD9834
   cs_on
   movlw	0x20			  ; bit13 = B28 = 1 = use two consecutive writes
;							  ;		to program the frequency: bits 13..0 first
   call		DDS.Byte
   movlw	0
   call		DDS.Byte
   cs_off
   return
; DDS.Init


DDS.Set.Freq	; set the frequency of the DDS
; enter with
; FSR2 = address of byte 0 of the 3 byte DDS frequency in Hz
; uses
;	  FSR0 and FSR1
;	  FSR2 is not modified
; NOTE: Frequencies are 4 byte values - the last byte is ignored!

   btfss	Flags, SET_DDS_FREQ ; skip if flag is set
   return
   btfsc	Flags, NOT_FREQUENCY ; skip if flag is clear
   return

   if USEPLL == 1
   bsf		OSCTUNE, PLLEN	  ; enable the PLL to get 64MHz for this function
   endif
; copy the frequency to Accumulator for math operations
   movff	FSR2L, FSR0L
   clrf		FSR0H
   lfsr		1, Accumulator
   call		Copy.Freq.Bytes
   clrf		INDF1			  ; clear byte 3 of Accumulator
   call		Mpy.10			  ; get 0.1Hz value

; copy the binary value
   lfsr		0, Accumulator	  ; point to LSB of final value location for Add.3
   lfsr		1, AccumTemp	  ; location of copy
   movlw	4
   call		Copy.Bytes
   lfsr		0, Accumulator	  ; point to LSB of final value location for Add.3
   lfsr		1, AccumTemp	  ; location of copy
; adjust Accumulator to account for 25MHz DDS oscillator
   call		Asr.3			  ; /2
   call		Asr.3			  ; /4
   call		Asr.3			  ; /8
   call		Asr.3			  ; /16
   call		Add.3			  ; A = A + A/16

   call		Asr.3			  ; /32
   call		Asr.3			  ; /64
   call		Asr.3			  ; /128
   call		Add.3			  ; A = A + A/16 + A/128

   call		Asr.3			  ; /256
   call		Asr.3			  ; /512
   call		Add.3			  ; A = A + A/16 + A/128 + A/512

   call		Asr.3			  ; /1024
   call		Add.3			  ; A = A + A/16 + A/128 + A/512 + A/1024

   call		Asr.3			  ; /2048
   call		Add.3			  ; A = A + A/16 + A/128 + A/512 + A/1024 + A/2048

   if USEPLL == 1
   bcf		OSCTUNE, PLLEN	  ; disable the PLL
   endif

;	  A = A( 1 + 1/16 + 1/128 + 1/512 + 1/1024 + 1/2048) =
;		  A ( 1 + (128+16+4+2+1)/2048 ) = A ( 1 + 151/2048 )
;		  A = 1.0737304 * (original value of Accumulator)
;   FDDS = (FClock/2^28) * N = .093132257 * N
;   if ASCII value = 100.0, then N = 1000*1.0737304
;   then FDDS = .093132257 * 1073.7304 = 99.9989Hz

;  The AD9834 requires that the frequency data be sent in the following order
;  	  lower 14 bits first: bits 13..8, bits 7..0
;  	  upper 14 bits next:  bits 27..22, bits 21..14
;  The uper two bits in the first byte of both pairs must contain the address
;  of the appropriate frequency register: this program uses only FREQ0:
;	  B15,B14 = 0,1

   cs_on
; send lower 14 bits
   movf		Accumulator+1, w  ; get bits 15..8 of frequency
   andlw	0x3F			  ; keep bits 13..8
   iorlw	0x40			  ; insert address for FREQ0 register
   call		DDS.Byte
   movf		Accumulator, w	  ; get bits 7..0
   call		DDS.Byte
; send upper 14 bits
   lfsr		0, Accumulator	  ; need to put
   call		Asl.3			  ;		bits 15 & 14
   call		Asl.3			  ;			  into byte 2
   movlw	0x40			  ; bits 27..22 = 0
   call		DDS.Byte		  ;		 and insert address for FREQ0 register
   movf		Accumulator+2, w  ; get bits 21..14
   call		DDS.Byte
   cs_off
   return
; DDS.Set.Freq


DDS.Byte	;														 8us
; send a byte to the DDS chip
; Enter with
;	  w = byte value to send

   movwf	SSPBUF			  ; transmit it
DDS.Byte.1
   btfss	PIR1, SSPIF		  ; is xmtr done?
   bra		DDS.Byte.1		  ; br if no
   bcf	 	PIR1, SSPIF		  ; show byte is done
   return
; DDS.Byte


IsDigit		; check if character is a digit						  15/21/13/16
;			   - decimal point is ignored!!
;  enter with:
;	  FSR2 = address of digit
;  exit with:
;	  C = 1 if character is a digit
;	  C = 0 if character is not a digit
;	  FSR0 incremented if character is a decimal point so next character can be
;			tested

   movlw	a'.'			  ; '.' = 0x2E
   cpfseq	INDF2			  ; continue processing if not a decimal point
   bra		IsDigit.1		  ; here if NOT '.'
; handle decimal point
   movf		POSTINC2, w		  ; point to next character
   bra		IsDigit			  ;		and try again
IsDigit.1
   movlw	a'0'-1
   cpfsgt	INDF2
   bra		IsDigit.no		  ; here if char < '0'
   movlw	a'9'+1
   cpfslt	INDF2
   bra		IsDigit.no		  ; here if char > '9'
   bsf		STATUS, C		  ; show char is a digit
   return
IsDigit.no
   bcf		STATUS, C		  ; show not a digit
   return
; IsDigit


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;						LCD functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; These functions utilize the four bit interface to an HD44780 compatible
; LCD controller.  The four data and two control signals do not have to be
; on the same I/O port.  IMPORTANT: the upper 4 bits of LCD_DATA are used as the
; data bus to the 44780
; Note: using PORTA as the data bus because
;	  PORTB has interrupt inputs needed by Quadrature Encoder
;	  PORTB has the interface bits for MPLAB
;	  PORTC has SPI port bits on both nibbles
;
; Main Functions:
;	 LCD.Print	  display a NULL terminated string
;		 enter with:
;			FSR0 = address of the first character
;
;	  LCD.Clear	  clear the display
;
;	  LCD.Line	  set cursor to the specified line
;		 enter with:
;			LCD_line_nbr = desired line number - 0 relative
;
;	  LCD.Pos	  set cursor to the specified line and column
;		 enter with
;			LCD_line_nbr = desired line number - 0 relative
;			LCD_col_nbr = desired column number - 0 relative
;
;	  LCD.PrintW		  print the character in the W register
;
; General Function used with LCD functions
;	  TxtROM.2.RAM  copy text from ROM to RAM
;		 enter with
;			FSR0 = address of text in ROM
;			text must be in low memory (12 bit address) and be NULL terminated
;		 FSR1 = address of destination RAM

; Use the following definitions to define the data port, control port and
; control bits:

LF_char	 equ   0x0B		; line feed
CR_char	 equ   0x0D		; carriage return
SP_char	 equ   0x20		; space
BSP_char equ   0x08		; back space


LCD.Init	 ; initialize the I/O ports for the LCD and set up the LCD

   movlw	0				  ; init outputs to 0
   movwf	LCD_DATA
   movlw	0x0F			  ; set up make bits 4-7 outputs
   movwf	LCD_DATA_TRIS

   bcf		LCD_CONTROL, LCD_ENA ; idle state for ENA
   bcf		LCD_CONTROL_TRIS, LCD_ENA ; set ENA as output
   bcf		LCD_CONTROL_TRIS, LCD_RS ; set RS as output
   bcf		LCD_CONTROL, LCD_RS ; set reg sel to cmd

; initialize the 44780 to 4 bit mode
; this sequence guarantees to set 4 bit mode without having to power down
; the LCD
   movlw	0x30			  ; 8 bit mode - wake up 1
   call		LCD.Nibble
   movlw	0x30			  ; 8 bit mode - wake up 2
   call		LCD.Nibble
   movlw	0x30			  ; 8 bit mode - wake up 3
   call		LCD.Nibble
   movlw	0x20			  ; 4 bit mode
   call		LCD.Nibble

   movlw	0x10			  ; set cursor
   call		LCD.Cmd
   movlw	0x0E			  ; Display on, blinking underline cursor
   call		LCD.Cmd
   movlw	0x06			  ; increment cursor after write
   call		LCD.Cmd

   delay	100				  ; wait 100 msec for LCD to respond
   call		LCD.Clear		  ; clear the display
Return.Inst
   return
; LCD.Init


LCD.Print	   ; display a string
; enter with:
;  FSR0 = address of the first character
;  the string must be null (\n) terminated

   movlw	SP_char-1		  ; test for special character
   cpfsgt	INDF0			  ; skip if NOT special character
   bra		LCD.Print.SP
   movf		POSTINC0, w		  ; get a character
   call		LCD.Data
   incf		LCD_col_nbr		  ; update column nbr
   bra		LCD.Print		  ; get next character
LCD.Print.Ex
   return

; handle special characters
LCD.Print.SP
   movlw	'\n'			  ; next line?
   cpfseq	INDF0	 		  ; skip if yes
   bra		LCD.Print.1		  ; try another character
; handle CR
   incf	 	LCD_line_nbr	  ; next line nbr
   call		LCD.Line
   bra		LCD.Print.z

LCD.Print.1
   movlw	'\b'	 		  ; back space?
   cpfseq	INDF0			  ; skip if yes
   bra		LCD.Print.2		  ; try another character
; handle back space
   decf		LCD_col_nbr		  ; back up one position
   call		LCD.Pos
   bra		LCD.Print.z

LCD.Print.2
   movlw	'\f'			  ; form feed (clear screen)?
   cpfseq	INDF0			  ; skip if yes
   bra		LCD.Print.3		  ; try another character
; handle form feed
   call		LCD.Clear		  ; clear the display and put cursor at 0,0
   bra		LCD.Print.z

LCD.Print.3
   movlw	'\r'			  ; start of current line?
   cpfseq	INDF0			  ; skip if yes
   bra		LCD.Print.z		  ; try another character
; handle CR
   clrf		LCD_col_nbr		  ; point to beginning of line
   call		LCD.Pos
   bra		LCD.Print.z

LCD.Print.z
   movf		POSTINC0, w		  ; remove current char and point to next char
   bz		LCD.Print.Ex	  ; br if null byte
   bra		LCD.Print
; LCD.Print


LCD.Clear
   movlw	0x01			  ; clear display
   call		LCD.Cmd
   clrf	 	LCD_col_nbr		  ; reset column
   clrf	 	LCD_line_nbr	  ;		and row values
   movlw	3				  ; delay for display
   goto		Delay.Wmsec		  ;		and return to caller
; LCD.Clear


LCD.Line		   ; set cursor to the specified line number, column 0
; enter with
;	  LCD_line_nbr = desired line number - 0 relative
; uses
;	  TBLPTR

   clrf		LCD_col_nbr		  ; set to beginning of line
; drop through to LCD.Pos
; LCD.Line


LCD.Pos		   ; set cursor to the specified line and column
; enter with
;	  LCD_line_nbr = desired line number - 0 relative
;	  LCD_col_nbr = desired column number - 0 relative
; uses
;	  TBLPTR

   movlw	LCD_LineOffset	  ; get address of offset table
   addwf	LCD_line_nbr, w	  ; add the line nbr
   movwf	TBLPTR
   clrf		TBLPTRH
   tblrd*+
   movf	 	TABLAT, w		  ; get the LCD address and command
   addwf	LCD_col_nbr, w	  ; insert column number
   goto		LCD.Cmd			  ; place the cursor
; LCD.Pos


LCD.PrintW		  ; print the character in the W register
   lfsr		0, LCD_line
   movwf	POSTINC0		  ; insert the character
   clrf		POSTDEC0		  ; followed by terminating null
   goto		LCD.Print		  ; print it and return
; LCD.PrintW


Erase.Char
   lfsr		0, Menu_erase	  ; delete the character at the cursor

;drop through to Print.ROM

Print.ROM	   ; print text from ROM
; enter with
;  FSR0 = address of text in ROM
;	  text must be in low memory (12 bit address) and be NULL terminated
; exit with
;	  bytes copied to LCD_line and printed
; uses
;	  FSR0 and FSR1

   lfsr		1, LCD_line
   call		Txt.ROM.2.RAM
   lfsr		0, LCD_line
   bra		LCD.Print		  ; print it and return
; Print.ROM


; Enter with the value in W for each of trhe next 3 functions
LCD.Cmd
   bcf		LCD_CONTROL, LCD_RS ; set reg sel to cmd
   bra		LCD.Byte

LCD.Data
   bsf		LCD_CONTROL, LCD_RS ; set reg sel to data
; drop through to LCD_Byte

LCD.Byte
; LCD_RS must be at the correct value before entering this function
; enter with W = byte value to send - upper nibble is sent first
   movwf	LCD_Value		  ; save value
   call		LCD.Nibble		  ; send upper 4 bits
   swapf	LCD_Value, w	  ; swap nibbles and retrieve value into w
   bra		LCD.Nibble		  ; send lower 4 bits and return
; LCD.Byte


LCD.Nibble
; enter with W<7:4> = 4 bit value

   andlw	0xF0			  ; keep only upper 4 bits
   movwf	Nibble			  ; save it
   movf	 	LCD_DATA, w		  ; get current port value
   andlw	0x0F			  ; keep lower 4 bits
   iorwf	Nibble, w		  ; merge with data value
   movwf	LCD_DATA		  ; write value to data port
; pulse the Ena bit
   bsf		LCD_CONTROL, LCD_ENA ; set the E bit
   call		Return.Inst		  ; 4 instruction cycles to widen the pulse
   bcf		LCD_CONTROL, LCD_ENA ; clear the E bit
   goto		Delay.100usec	  ; delay and return
; LCD.Nibble


Txt.ROM.2.RAM		   ; copy text from ROM to RAM
; enter with
;  FSR0 = address of text in ROM
;	  text must be in low memory (12 bit address) and be NULL terminated
;  FSR1 = address of destination RAM
; exit with
;	  bytes copied
;	  FSR1 modified

   movf		FSR0L, w		  ; get low byte of address
   movwf	TBLPTR
   movf		FSR0H, w		  ; get high byte of address
   movwf	TBLPTRH
   clrf		TBLPTRU
Txt.ROM.2.RAM.1
   tblrd*+					  ; get a byte from ROM
   movf	 	TABLAT, w		  ; copy it
   movwf	POSTINC1		  ; save it
   bnz		Txt.ROM.2.RAM.1	  ; until NULL encountered
   return
; Txt.ROM.2.RAM


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;						end LCD functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	  Delay functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Delay.W100msec	   ; delay nbr of 100 msec intervals in W - up to 256 (w=0)
; uses W, DelayCount_3

   movwf	DelayCount_3	  ; save nbr of 100 msec intervals
Delay.W100msec.1
   delay	100
   decf		DelayCount_3	  ; done all 100 msec intervals?
   bnz		Delay.W100msec.1  ; br if no
   return


Delay.Wmsec	   ; delay nbr of msec in W - up to 255
; uses W, DelayCount_2

   movwf	DelayCount_2	  ; get nbr of msec
   andwf	DelayCount_2	  ; is delay 0?
   bz		Delay.Wmsec.2	  ; br if yes
Delay.Wmsec.1
   call		Delay.Msec		  ; delay 1msec
   decf		DelayCount_2	  ; update msec counter
   bnz		Delay.Wmsec.1	  ; br if not done
Delay.Wmsec.2
   return
; Delay.Wmsec


Delay.Msec	   ; delay one millisecond
; uses W and DelayCount_1

   movlw	9				  ; 2
   movwf	DelayCount_1	  ; 2
Delay.Msec.1
   call		Delay.100usec	  ; Exactly 100usec
   decf		DelayCount_1		  ; 1
   bnz		Delay.Msec.1	  ; 2(1)
; loop time = (n*100usec) + ( (n-1)*3 + 2 )cycles = 900usec + 26cycles

   movlw	90				  ; 1 attempted adjustment for loop execution
   call		Delay.100usec.1	  ; 2+[4(n-1)+3+5] = 2+4n-4+8 = 4n+6
   return					  ; 2
; time of complete function - including the calling statement =
;	  900usec +[2 + 4 + 26 + (4n+6) + 2]cycles = 900usec + (4n+40) cycles
;	  therefore: 4n+38 == 400,  n = (400-40)/4 = 90
; Delay.Msec


Delay.100usec	  ; delay one hundred micro seconds - exactly!
; this routine requires a 16MHz clock - instruction cycle time - 250ns
; requires 400 instruction cycles
; uses W, Delay_100val
; total execution time = (2) + 2 + 4(n-1) + 3 + 5 = 4+4(97)+8 = 388+12 = 400
; the (2) accounts for the CALL instruction in the calling function
   movlw	98				  ; 1
Delay.100usec.1				  ; enter here with a different value for W
   movwf	Delay_100val	  ; 1
Delay.100usec.2				  ; total loop time is 4(n-1) + 3
   decf	 	Delay_100val	  ; 1 update inner loop count
   nop						  ; 1 - to make loop time a "nice" value == 4
   bnz		Delay.100usec.2	  ; 2(1)  br if not done
   nop						  ; 1 to make
   nop						  ; 1	exactly
   nop						  ; 1	   	400 cycles
   return					  ; 2
; Delay.100usec


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	  Initialize the I/O ports
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Init.Ports

; Port A
   CLRF		ANSEL			  ; do not use PORTA bits for the A/D
   CLRF		PORTA			  ; Initialize PORT by clearing output data latches
   MOVLW	0x0F	 		  ; Set RA<7:4> as outputs,
   MOVWF	TRISA			  ;		RA<3:0> as inputs

; Port B
   CLRF		PORTB			  ; Initialize PORT by clearing output data latches
   CLRF		ANSELH			  ; Set RB<4:0> as digital I/O pins
							  ;		(required if config bit PBADEN is set)
   MOVLW	b'11100111'		  ; Set RB<7:5> as inputs,
   MOVWF	TRISB			  ;		RB<4:3> as outputs,  RB<2:0> as inputs
   movlw	b'00100111'		  ; enable weak pull-ups
   movwf	WPUB			  ;		on bits 5, 2..0
   clrf		ANSELH			  ; make sure Port B is NOT used for the A/D

; Port C
   clrf		PORTC			  ; Initialize PORT by clearing output data latches
   clrf	 	TRISC			  ; all bits are output
   bsf		LED_PORT, LED0	  ; turn on LED 0

; other system registers
   movlw	0x78			  ; 16MHz, fosc<3:0>
   movwf	OSCCON			  ; set up oscillator control to 16MHz
   clrf		RCON			  ; clear some interrupt flags, IPEN=0 <7>
   movlw	b'10010000'		  ; GIE=1, INT0IE=1
   movwf	INTCON			  ; global enable interrupts and RB0 interrupt
   movlw	b'01000000'		  ; INTEDG0 = 1
   movwf	INTCON2			  ; Ext Int 0 on rising edge
   clrf	 	INTCON3			  ; disable other ext interrupt inputs
   clrf	 	PIE1			  ; disable
   clrf	 	PIE2			  ;		peripheral interrupts
   return
; Init.Ports


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;	  Interrupt routines
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

LoPriInt ; Lo pri ints cannot use fast retfie if interrupt priority is enabled

   retfie

HiPriInt
   movff	STATUS, STATUS_save
   movwf	W_save

; only interrupt should be INT0 on RB0
   btfss	INTCON, INT0IF	  ; exit if not RB0 interrupt
   bra		HiPriIntExit
; handle RB0 interrupt
   bcf		INTCON, INT0IF	  ; clear the interrupt
   btfsc	QE_PORT, QE_Q	  ; skip if QD-Q is low
   bra		HiPriInt.1
   incf		QD_count
   bra		HiPriIntExit
HiPriInt.1
   decf		QD_count
HiPriIntExit

   movf		W_save, w
   movff	STATUS_save, STATUS

   retfie	FAST			  ; fast return: restores STATUS, W and BSR
; HiPriInt

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;						Math functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Mpy.10		; multiply a 3-byte value by 10
; enter with:
;	  Accumulator = value to multiply
; exit with:
;	  Accumulator updated
; uses:
;	  FSR0 and FSR1

   lfsr		0, Accumulator+2
   lfsr		1, AccumTemp+2	  ; address to put copy
; the following four instructions leave FSR0 and FSR1 pointing to the LSBs
   movff	POSTDEC0, POSTDEC1 ; copy
   movff	POSTDEC0, POSTDEC1 ;	current value of
   movff	INDF0, INDF1	  ;			  Accumulator
   call		Asl.3			  ; x2
   call		Asl.3			  ;		x4
   call		Add.3			  ;		   x5
; drop through to Asl.3 and return		  x10
; Mpy.10

; **** DO NOT MOVE Asl.3 - it MUST follow Mpy.10 ****

Asl.3  ;  arithmetic shift of 3 bytes left by 1 bit
; this is a multiply by two
; the shift is performed in-place, starting with the LSB
; enter with:
;	  FSR0 = address of LSB
; exit with:
;	  FSR0 = address of LSB
; both the C and Z bits are result of the last shift operation

   bcf		STATUS, C		  ; ensure LSbit is 0
   rlcf		POSTINC0
   rlcf		POSTINC0
   rlcf		INDF0
; point back to LSByte
   mulwf	POSTDEC0		  ; mulwf does not affect STATUS
   mulwf	POSTDEC0		  ;	 and leaves the product in PRODH:PRODL
   return
; Asl.3


Asr.3  ;  arithmetic shift of 3 bytes right by 1 bit
; this is a divide by two
; the shift is performed in-place, starting with the MSB
; enter with:
;	  FSR1 = address of LSB
; exit with:
;	  FSR1 = address of LSB
; both the C and Z bits are the result of the last shift operation

   mulwf	POSTINC1		  ; use mulwf
   mulwf	POSTINC1		  ;		to point to MSByte
   bcf		STATUS, C		  ; ensure MSbit is 0
   rrcf		POSTDEC1
   rrcf		POSTDEC1
   rrcf		INDF1
   return
; Asr.3


Add.3	 ; add two 3-Byte positive integers							 18
; Enter with
;	  FSR0 = address of LSB of value 1
;	  FSR1 = address of LSB of value 2
; Exit with
;	  (FSR0) = (FSR0) + (FSR1)
; Note: both FSR0 and FSR1 are used but restored to original values

; update byte 0 and point to byte 1
   movf		POSTINC1, W		  ; get byte 0 of value 2
   addwf	POSTINC0		  ; 1st add must not use C bit
; update byte 1 and point to byte 2
   movf		POSTINC1, W		  ; get byte 1 of value 2
   addwfc	POSTINC0		  ; update using carry
; update byte 2 and point to byte 1
   movf		POSTDEC1, W		  ; get byte 2 of value 2
   addwfc	POSTDEC0 		  ; update using carry
; set both pointers bck to byte 0
   mulwf	POSTDEC1		  ; use mulwf to point back to byte 0
   mulwf	POSTDEC0		  ; use mulwf to point back to byte 0
   return
; Add.3


Sub.3	 ; subtract two 3-Byte positive integers							 18
; Enter with
;	  FSR0 = address of LSB of value 1
;	  FSR1 = address of LSB of value 2
; Exit with
;	  (FSR0) = (FSR0) - (FSR1)
; Note: both FSR0 and FSR1 are used but restored to original values
; Note: a negative result will be changed to 0

; update byte 0 and point to byte 1
   movf		POSTINC1, W		  ; get byte 0 of value 2
   subwf	POSTINC0		  ; 1st sub must not use C bit
; update byte 1 and point to byte 2
   movf		POSTINC1, W		  ; get byte 1 of value 2
   subwfb	POSTINC0		  ; update using carry
; update byte 2
   movf		POSTDEC1, W		  ; get byte 2 of value 2 and point to byte 1
   subwfb	INDF0	  		  ; update using carry
; test to ensure non-negative result
   mulwf	POSTDEC1		  ; use mulwf to point back to byte 0
   movf		INDF0, W		  ; use to set/clr N bit
   bn		Sub.3.10			  ; br if negative
; set both pointers bck to byte 0
   mulwf	POSTDEC0		  ; use mulwf to point back to byte 1
   mulwf	POSTDEC0		  ; use mulwf to point back to byte 0
   return
; handle negative result
Sub.3.10
   clrf		POSTDEC0		  ; clear byte 2 and point to byte 1
   clrf		POSTDEC0		  ; clear byte 1 and point to byte 0
   clrf		INDF0	 		  ; clear byte 0
   return
; Sub.3


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;		 divide 24 bit integer by 16 bit integer
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; the following divide function was copied from:
;		 http://techref.massmind.org/techref/microchip/math/div/24by16.htm
;Inputs:
;   Dividend - Accumulator where Accumulator[0] = LSByte
;   Divisor  - AccumTemp where AccumTemp[0] = LSByte
;Temporary:
;   Counter  - LoopCount
;Output:
;   Quotient - Accumulator
;   Remainder- Addend where Addend[0] = LSByte

FXD2416U
   CLRF		Addend+1
   CLRF		Addend
   MOVLW	24
   MOVWF	LoopCount
LOOPU2416
   RLcF		Accumulator, F	  ;shift dividend left to move next bit to remainder
   RLcF		Accumulator+1, F  ;and shift in next bit of result
   RLcF		Accumulator+2, F  ;

   RLcF		Addend, F		  ;shift carry (next dividend bit) into remainder
   RLcF		Addend+1, F

   RLcF		LoopCount, F	  ;save carry in counter, since remainder
							  ;can be 17 bit long in some cases (e.g.
							  ;0x800000/0xFFFF)

   MOVF		AccumTemp, W	  ;substract divisor from 16-bit remainder
   SUBWF	Addend, F		  ;
   MOVF		AccumTemp+1, W	  ;
   BTFSS	STATUS, C		  ;
   INCFSZ	AccumTemp+1, W	  ;
   SUBWF	Addend+1, F		  ;

;here we also need to take into account the 17th bit of remainder, which
;is in LoopCount.0. If we don't have a borrow after subtracting from lower
;16 bits of remainder, then there is no borrow regardless of 17th bit
;value. But, if we have the borrow, then that will depend on 17th bit
;value. If it is 1, then no final borrow will occur. If it is 0, borrow
;will occur.

   bnc	    FXD2416U.1
   BSF		LoopCount, 0	  ;then no no borrow in result. Overwrite
							  ;LoopCount.0 with 1 to indicate no
							  ;borrow.
							  ;if borrow did occur, LoopCount.0 will
							  ;hold the eventual borrow value (0-borrow,
							  ;1-no borrow)
FXD2416U.1
   BTFSC	LoopCount, 0	  ;if no borrow after 17-bit subtraction
   GOTO		UOK46LL			  ;skip remainder restoration.

   ADDWF	Addend+1, F		  ;restore higher byte of remainder. (w
							  ;contains the value subtracted from it
							  ;previously)
   MOVF		AccumTemp, W	  ;restore lower byte of remainder
   ADDWF	Addend, F		  ;

UOK46LL
   bcf		STATUS, C		  ;copy bit LoopCount.0 to carry
   RRcF		LoopCount, F	  ;and restore counter

   DECFSZ	LoopCount, f	  ;decrement counter
   bra		LOOPU2416		  ;and repeat loop if not zero. carry
							  ;contains next quotient bit (if borrow,
							  ;it is 0, if not, it is 1).

   RLcF		Accumulator, F	  ;shift in last bit of quotient
   RLcF		Accumulator+1, F
   RLcF		Accumulator+2, F
   RETURN
; FXD2416U


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;						EEPROM functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

Read.EEPROM	   ; read data from the EEPROM
; These values will be stored into non-access RAM in RAM block 0
; uses:
;	  FSR0
; note: since FSR0 is the destination, the BSR is not needed

   clrf		EEADR			  ; data starts at EEPROM address = 0
   clrf		EECON1			  ; set up to read EEPROM
   lfsr		0, EEPROM_data	  ; get address of start of EEPROM RAM data

; check for valid data - value in first byte should be = EEPROM_SIZE
   bsf		EECON1, RD		  ; read a byte
   movff	EEDATA, INDF0	  ; store it
   movlw	EEPROM_SIZE		  ; nbr of bytes
   subwf	POSTINC0, w		  ; is value correct?
   bnz		Read.EEPROM.20	  ; br if no

   movlw	EEPROM_SIZE-1	  ; get number of bytes remaining to read
   movwf	LoopCount		  ; save byte count
   incf		EEADR			  ; point to next EEPROM address
Read.EEPROM.10
   bsf		EECON1, RD		  ; read a byte
   movff	EEDATA, POSTINC0  ; store it
   incf		EEADR			  ; point to next EEPROM address
   decfsz	LoopCount		  ; update byte count and exit if done (=0)
   bra		Read.EEPROM.10	  ; br if not done all bytes
Read.EEPROM.20
   return
; Read.EEPROM


Write.EEPROM	   ; write data to the EEPROM
; These values are read from non-access RAM in block 0
; uses:
;	  FSR2
; note: since FSR2 is the source, the BSR is not needed

   bcf		PIE2, EEIE		  ; ensure EEPROM interrupt is disabled
   clrf		EEADR			  ; data starts at EEPROM address = 0
   clrf		EECON1			  ; set up to read EEPROM
   lfsr		2, EEPROM_data	  ; get address of start of EEPROM data in RAM
   movlw	EEPROM_SIZE		  ; get number of bytes to write
   movwf	LoopCount		  ; save working copy of byte count for write loop
   movwf	DDS_Flag		  ; show valid data in EEPROM
   bsf		EECON1, WREN	  ; enable EEPROM writes
Write.EEPROM.5
; set up to write a byte
   movff	POSTINC2, EEDATA  ; put byte into EEPROM buffer
   bcf		INTCON, GIE		  ; disable interrupts
   movlw	0x55			  ; sequence
   movwf	EECON2			  ;		required
   movlw	0xAA			  ;		   for writing
   movwf	EECON2			  ;			  to EEPROM
   bsf		EECON1, WR		  ; begin write
   bsf		INTCON, GIE		  ; enable interrupts
Write.EEPROM.10
; wait for byte to finish
   btfsc	EECON1, WR		  ; skip if done
   bra		Write.EEPROM.10
   bcf	 	PIR2, EEIF		  ; clear the EEPROM interrupt bit
; update address and byte count
   incf		EEADR			  ; point to next EEPROM address
   decfsz	LoopCount		  ; update and exit if done (LoopCount=0)
   bra		Write.EEPROM.5	  ; br if not done all bytes
   bcf		EECON1, WREN	  ; disable writes

; verify EEPROM contents
Write.EEPROM.20
   movlw	3				  ; set up to put the
   movwf	LCD_line_nbr	  ;		cursor on line 3
   lfsr		0, TxtSaveVerify
   call		Print.ROM
   clrf		EEADR			  ; data starts at EEPROM address = 0
   clrf		EECON1			  ; set up to read EEPROM
   lfsr		2, EEPROM_data	  ; get address of start of EEPROM RAM data
   movlw	EEPROM_SIZE		  ; get number of bytes to read
   movwf	LoopCount		  ; save working copy of byte count for read loop
Write.EEPROM.30
   bsf		EECON1, RD		  ; command to read a byte
   movf		EEDATA, W		  ; get it
   cpfseq	POSTINC2		  ; compare to data in RAM
   bra		Write.EEPROM.40	  ; br if error
   incf		EEADR			  ; point to next EEPROM address
   decfsz	LoopCount		  ; update byte count and exit if done (=0)
   bra		Write.EEPROM.30	  ; br if not done all bytes
   lfsr		0, TxtSaveSuccess
Write.EEPROM.35
   call		Print.ROM		  ; print success/error message
   movlw	30				  ; set up for 3 second
   goto		Delay.W100msec	  ;		delay and return to caller
Write.EEPROM.40
   lfsr		0, TxtSaveError
   bra		Write.EEPROM.35

                END
